<?php
/*
Plugin Name: Sitemap Generator for WordPress & WooCommerce
Plugin URI: https://installments.ge
Description: Automatically generates sitemaps for posts, pages, and WooCommerce products.
Version: 1.0
Author: installments.ge
Author URI: https://installments.ge
License: GPL2
*/

if (!defined('ABSPATH')) exit;

class WP_Woo_Sitemap_Generator {
    private $sitemap_dir;

    public function __construct() {
        $this->sitemap_dir = ABSPATH;
        add_action('publish_post', [$this, 'generate_sitemap']);
        add_action('publish_page', [$this, 'generate_sitemap']);
        add_action('save_post', [$this, 'generate_sitemap']);
        add_action('woocommerce_update_product', [$this, 'generate_sitemap']);
        add_action('admin_menu', [$this, 'add_admin_page']);
    }

    public function generate_sitemap() {
        $sitemaps = [
            'sitemap.xml' => $this->get_sitemap_index(),
            'sitemap-posts.xml' => $this->get_posts_sitemap(),
            'sitemap-products.xml' => $this->get_products_sitemap()
        ];
        
        foreach ($sitemaps as $filename => $content) {
            file_put_contents($this->sitemap_dir . $filename, $content);
        }
    }

    private function get_sitemap_index() {
        return "<?xml version='1.0' encoding='UTF-8'?>\n<sitemapindex xmlns='http://www.sitemaps.org/schemas/sitemap/0.9'>\n" .
            "<sitemap><loc>" . site_url('/sitemap-posts.xml') . "</loc></sitemap>\n" .
            "<sitemap><loc>" . site_url('/sitemap-products.xml') . "</loc></sitemap>\n" .
            "</sitemapindex>";
    }

    private function get_posts_sitemap() {
        $posts = get_posts(['post_type' => 'post', 'numberposts' => -1, 'post_status' => 'publish']);
        $xml = "<?xml version='1.0' encoding='UTF-8'?>\n<urlset xmlns='http://www.sitemaps.org/schemas/sitemap/0.9'>\n";
        
        foreach ($posts as $post) {
            $xml .= "<url><loc>" . get_permalink($post->ID) . "</loc></url>\n";
        }
        
        $xml .= "</urlset>";
        return $xml;
    }

    private function get_products_sitemap() {
        if (!class_exists('WooCommerce')) return '';
        
        $products = get_posts(['post_type' => 'product', 'numberposts' => -1, 'post_status' => 'publish']);
        $xml = "<?xml version='1.0' encoding='UTF-8'?>\n<urlset xmlns='http://www.sitemaps.org/schemas/sitemap/0.9'>\n";
        
        foreach ($products as $product) {
            $xml .= "<url><loc>" . get_permalink($product->ID) . "</loc></url>\n";
        }
        
        $xml .= "</urlset>";
        return $xml;
    }

    public function add_admin_page() {
        add_options_page('Sitemap Generator', 'Sitemap Generator', 'manage_options', 'sitemap-generator', [$this, 'admin_page_content']);
    }

    public function admin_page_content() {
        echo "<div class='wrap'><h1>Sitemap Generator</h1>";
        echo "<p>Sitemaps are generated automatically. You can manually regenerate them if needed.</p>";
        echo "<form method='post'><button name='generate_sitemap' class='button button-primary'>Generate Sitemap</button></form>";
        
        if (isset($_POST['generate_sitemap'])) {
            $this->generate_sitemap();
            echo "<p><strong>Sitemap regenerated successfully!</strong></p>";
        }
        
        echo "<h2>Download Sitemaps</h2>";
        echo "<p><a href='" . site_url('/sitemap.xml') . "' class='button button-secondary' download>Download Main Sitemap</a></p>";
        echo "<p><a href='" . site_url('/sitemap-posts.xml') . "' class='button button-secondary' download>Download Posts Sitemap</a></p>";
        echo "<p><a href='" . site_url('/sitemap-products.xml') . "' class='button button-secondary' download>Download Products Sitemap</a></p>";
        
        echo "</div>";
    }
}

new WP_Woo_Sitemap_Generator();
